; Fixed _ClipBoard_SetData  (to be consistent with new _ClipBoard_GetData function changes for v3.3.1.2)

; #FUNCTION# ====================================================================================================================
; Name...........: _ClipBoard_SetData
; Description ...: Places data on the clipboard in a specified clipboard format
; Syntax.........: _ClipBoard_SetData($vData[, $iFormat = 1])
; Parameters ....: $vData     - Data in Binary or String format, or optionally NULL (0) (owner must render, see below).
;                  +IMPORTANT: If a String is passed, and it is not of type $CF_TEXT, $CF_OEMTEXT, or $CF_UNICODETEXT,
;                  + it will be treated as an ANSI string. To force Unicode strings for other types, you must pass the data
;                  + in Binary format. Also, do NOT pass $CF_UNICODETEXT in Binary format (causes garbled data).
;                  +When $vData is NULL, it indicates that the window provides data in the specified clipboard format upon request.
;                  +If a window delays rendering, it must process the $WM_RENDERFORMAT and $WM_RENDERALLFORMATS messages.  If this
;                  +function succeeds, the system owns the object identified by the $hMemory parameter.  The application may not
;                  +write to or free the data once ownership has been transferred to the system, but it can lock and read from the
;                  +data until the _ClipBoard_Close function is called.  The memory must be unlocked before the clipboard is
;                  +closed.  If the $hMemory parameter identifies a memory object, the object must have been allocated using the
;                  +function with the $GMEM_MOVEABLE
;                  +flag.
;                  $iFormat     - Specifies a clipboard format:
;                  |$CF_TEXT            - Text format
;                  |$CF_BITMAP          - Handle to a bitmap (HBITMAP)
;                  |$CF_METAFILEPICT    - Handle to a metafile picture (METAFILEPICT)
;                  |$CF_SYLK            - Microsoft Symbolic Link (SYLK) format
;                  |$CF_DIF             - Software Arts' Data Interchange Format
;                  |$CF_TIFF            - Tagged image file format
;                  |$CF_OEMTEXT         - Text format containing characters in the OEM character set
;                  |$CF_DIB             - BITMAPINFO structure followed by the bitmap bits
;                  |$CF_PALETTE         - Handle to a color palette
;                  |$CF_PENDATA         - Data for the pen extensions to Pen Computing
;                  |$CF_RIFF            - Represents audio data in RIFF format
;                  |$CF_WAVE            - Represents audio data in WAVE format
;                  |$CF_UNICODETEXT     - Unicode text format
;                  |$CF_ENHMETAFILE     - Handle to an enhanced metafile (HENHMETAFILE)
;                  |$CF_HDROP           - Handle to type HDROP that identifies a list of files
;                  |$CF_LOCALE          - Handle to the locale identifier associated with text in the clipboard
;                  |$CF_DIBV5           - BITMAPV5HEADER structure followed by bitmap color and the bitmap bits
;                  |$CF_OWNERDISPLAY    - Owner display format
;                  |$CF_DSPTEXT         - Text display format associated with a private format
;                  |$CF_DSPBITMAP       - Bitmap display format associated with a private format
;                  |$CF_DSPMETAFILEPICT - Metafile picture display format associated with a private format
;                  |$CF_DSPENHMETAFILE  - Enhanced metafile display format associated with a private format
; Return values .: Success      - The handle to the data
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Modified.......:
; Remarks .......: This function performs all of the steps neccesary to put data on the clipboard.  It will allocate the global
;                  memory object, open the clipboard, place the data on the clipboard and close the clipboard.  If you need more
;                  control over putting data on the clipboard, you may want to use the _ClipBoard_SetDataEx function.
; Related .......: _ClipBoard_GetData, _ClipBoard_SetDataEx
; Link ..........;
; Example .......; Yes
; ===============================================================================================================================
Func _ClipBoard_SetData($vData, $iFormat = 1)
	Local $tData, $hLock, $hMemory, $iSize

	; Special NULL case? (the option to provide clipboard formats upon request)
	If IsNumber($vData) And $vData = 0 Then
		; No need to allocate/set memory
		$hMemory = $vData
	Else		
		; Test if the format is Binary or String format (only supported formats)
		If IsBinary($vData) Then
			$iSize = BinaryLen($vData)	
		ElseIf IsString($vData) Then
			$iSize = StringLen($vData)
		Else
			; Unsupported data type
			Return SetError(2,0,0)
		Endif
		$iSize+=1
		
		; Memory allocation is in bytes, yet Unicode text is 2-bytes wide
		If $iFormat = $CF_UNICODETEXT Then
			; Multiply $iSize (Character length for Unicode text) by 2 for Unicode
			$hMemory = _MemGlobalAlloc($iSize * 2, $GHND)
		Else
			$hMemory = _MemGlobalAlloc($iSize, $GHND)
		EndIf
		
		If $hMemory = 0 Then Return SetError(-1, 0, 0)
		$hLock = _MemGlobalLock($hMemory)
		If $hLock = 0 Then Return SetError(-2, 0, 0)
	
		Switch $iFormat
			Case $CF_TEXT, $CF_OEMTEXT
				$tData = DllStructCreate("char[" & $iSize & "]", $hLock)
			Case $CF_UNICODETEXT
				$tData = DllStructCreate("wchar[" & $iSize & "]", $hLock)
			Case Else
				; Every other type is treated as Binary, or ASCII Strings
				$tData = DllStructCreate("ubyte[" & $iSize & "]", $hLock)
		EndSwitch

		DllStructSetData($tData, 1, $vData)
		_MemGlobalUnlock($hMemory)
	EndIf
	
	If Not _ClipBoard_Open(0) Then Return SetError(-5, 0, 0)
	If Not _ClipBoard_Empty() Then Return SetError(-6, 0, 0)
	If Not _ClipBoard_SetDataEx($hMemory, $iFormat) Then
		_ClipBoard_Close()
		Return SetError(-7, 0, 0)
	EndIf

	_ClipBoard_Close()
	Return $hMemory
EndFunc   ;==>_ClipBoard_SetData
